#ifndef _WIN32
#include <unistd.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>

#include "libfma.h"
#include "lf_internal.h"
#include "lf_fabric.h"
#include "lf_switch.h"
#include "lf_product_def.h"
#include "lf_wget.h"

/*
 * Local data structures
 */
struct sw32_encl_desc {
  char *product_id;
  int bp_type;
};
static struct sw32_encl_desc sw32_encl_info[] = {
  { "M3-CLOS-ENCL", 1 },
  { "M3-SPINE-ENCL", 2 },
  { "10G-SW16-16C", 3},
  { "10G-CLOS-ENCL", 4},
  { NULL, 0 }
};

/*
 * local functions
 */
static int note_new_x32_linecard(FILE *fp, struct lf_enclosure *ep, int slot,
                                 char *product_id);


/* helper macro for loop below - temp variables used because some compilers
 * hate the hardcoded strcmp(NULL, wp[1]); that can result.
 */
#define PARSE_VALUE(D, S0, S1, S2, VI, V)			\
  else if (strcmp(wp[0], (S0)) == 0) {				\
    void *__s1 = (S1);						\
    void *__s2 = (S2);						\
    if ((__s1 == NULL || strcmp(__s1, wp[1]) == 0)		\
         && (__s2 == NULL || strcmp(__s2, wp[2]) == 0)) {	\
      if ((D) == NULL) {					\
	badstr = (S0);						\
      } else {							\
	(D)->new_vals.V = atoll(wp[VI]);			\
      }								\
    }								\
  }

int
lf_query_switch_x32_html(
  lf_enclosure_t *ep)
{
  FILE *fp;
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  int rc;
  int wc;
  int userslot;
  int slot;
  int port;
  int xbar;
  int port_is_xbar;
  struct lf_linecard *lp;
  struct lf_linecard_data *ldp;
  struct lf_xbarport_data *xdp;
  struct lf_xcvr_data *xcdp;
  struct lf_xbar *xp;
  int temp_index;
  char *badstr;

  /* open and read all switch components */
  fp = lf_wget_open(ep->name, "/cgi/web.cgi?all");
  if (fp == NULL) {
    LF_ERROR(("opening stream to %s", ep->name));
  }

  /*
   * Initialize some things
   */
  xbar = -1;
  port = -1;
  lp = NULL;
  ldp = NULL;
  userslot = -1;
  slot = -1;
  port_is_xbar = 0;
  xdp = NULL;
  xcdp = NULL;
  xp = NULL;
  temp_index = -1;

  /* 
   * Read the whole file.  This approach to parsing the stream just remembers
   * the last slot, port number, and port type to know what to do with data
   */
  while (lf_wget_next_line(fp, buf, wp, &wc) != NULL) {
    badstr = NULL;

    /* this first batch of words are the control words */

    /* remember last slot number */
    if (strcmp(wp[0], "slot") == 0 && wc == 3) {

      /* skip non-numeric slots */
      if (!isdigit(wp[1][0])) continue;

      userslot = atoi(wp[1]);
      if (userslot >= ep->lc_slotbase
	  && userslot < ep->lc_slotbase + ep->num_lc_slots) {
	slot = userslot - ep->lc_slotbase;
	lp = ep->slots[slot];

	if (lp != NULL) {
	  ldp = lp->data;
	  ldp->seen = TRUE;	/* mark this slot as seen */

	/* If lp NULL, this is an unexpected card!  Note the change. */
	} else {
	  rc = note_new_x32_linecard(fp, ep, slot, wp[2]);
	  if (rc == -1) {
	    LF_ERROR(("error noticing new linecard in %s[%d]",
		ep->name, userslot));
	  }
	}

      } else if (userslot >= ep->bp_slotbase
	         && userslot < ep->bp_slotbase + ep->num_bp_slots) {
	slot = userslot - ep->bp_slotbase + ep->num_lc_slots;
	lp = ep->slots[slot];
	if (lp != NULL) {
	  ldp = lp->data;
	  ldp->seen = TRUE;	/* mark this slot as seen */
#if 0 /* permit backplane slots to be discovered */
	} else {
	  LF_ERROR(("unexpected backplane slot in %s[%d]", ep->name, userslot));
#endif
	}

      } else {
	lp = NULL;
	ldp = NULL;
      }

    } else if (lp == NULL) {	/* all else needs a linecard */
      continue;

    /* got serial number and product id */
    } else if (wc > 2
	&& strcmp(wp[0], "serial") == 0
	&& strcmp(wp[1], "number") == 0) {
      char *serial_no;
      char *product_id;

      serial_no = wp[2];
      product_id = (wc > 3) ? lf_product_id_alias(wp[3]) : NULL;

      /*
       * This is the time to check for a new or changed  linecard.
       * slot is used to keep from doing this twice since serial number can
       * be in the stream twice
       */
      if (slot >= 0) {

	/* if no lp, but slot is good, found a new linecard */
	if (lp == NULL) {
	  rc = lf_note_new_linecard(ep, slot, product_id, serial_no);
	  if (rc == -1) {
	    LF_ERROR(("error noticing new linecard in %s[%d]",
		       ep->name, userslot));
	  }

	/*
	 * lp is good- First, fill in serial no if it is missing, then if
	 * either serial_no or product id don't match, it's a different card
	 */
	} else {

	  /*if no serial number, set it */
	  if (lp->serial_no == NULL || lp->serial_no[0] == '\0') {
	    LF_FREE(lp->serial_no);
	    LF_DUP_STRING(lp->serial_no, serial_no);
	  }

	  /* if serial number or product ID different, card changed */
	  if (strcmp(lp->serial_no, serial_no) != 0
	      || (product_id != NULL
		  && strcmp(lp->product_id, product_id) != 0)) {
	  
	    rc = lf_note_new_linecard(ep, lp->slot, product_id, serial_no);
	    if (rc == -1) {
	      LF_ERROR(("error noticing changed linecard in %s[%d]",
			 ep->name, userslot));
	    }
	  }
	}
	slot = -1;	/* serial number shows up twice - who knows why?? */
      }

    /* we are now looking at xbar ports */
    } else if (strcmp(wp[0], "xbar32") == 0) {
      port_is_xbar = TRUE;
      xbar = atoi(wp[1]);
      if (xbar < 0 || xbar >= lp->num_xbars) {
	LF_ERROR(("unexpected xbar %d on slot %d", xbar, userslot));
      }
      xp = LF_XBAR(lp->xbars[xbar]);

    /* we are now looking at transceiver ports */
    } else if (lp->num_xcvrs > 0 
               && strcmp(wp[0], lp->def->xcvr_port_string) == 0) {
      int port_index;
      port_is_xbar = FALSE;
      port = atoi(wp[1]);

#define MON_FW_WITH_CORRECT_PORT_LABELS 1
#if MON_FW_WITH_CORRECT_PORT_LABELS
      {
        for (port_index=0; port_index<lp->num_xcvrs; ++port_index) {
          /* find the right port given the label */
          if (lp->xcvr_labels[port_index] == port) {
            break;
          }
        }
        if (port_index >= lp->num_xcvrs) {
	  LF_ERROR(("bad transceiver number %d, slot=%d, prod=%s, upgrade switch firmware?",
	    port, lf_slot_display_no(lp), lp->product_id));
	}
        port = port_index;
      }
#else
     if (port < 0 || port >= lp->num_xcvrs) {
       LF_ERROR(("bad transceiver number %d", port));
     }
#endif

      xcdp = LF_XCVR(lp->xcvrs[port])->data;

    /* xbar port number */
    } else if (strcmp(wp[0], "xbar32_port") == 0) {
      port = atoi(wp[1]);
      if (!port_is_xbar) {
	LF_ERROR(("Unexpected xbar port %d", port));
      }
      if (port < 0 || port >= xp->num_ports) {
	LF_ERROR(("bad xbar port number %d", port));
      }
      xdp = xp->data + port;

    /* save xbar ID if not known */
    } else if (strcmp(wp[0], "id") == 0) {
      if (xp == NULL) {
        LF_ERROR(("Unexpected id tag"));
      }
      xp->xbar_id = atoi(wp[1]);


    /* get quadrant disable value */
    } else if (strcmp(wp[0], "quadrant") == 0) {
      if (xp == NULL) {
        LF_ERROR(("Unexpected quadrant_disable value"));
      }
      if (wc > 2) {
	if (strcmp(wp[2], "on") == 0) {
	  xp->xbar_data->new_vals.quadrant_disable = 1;
	} else {
	  xp->xbar_data->new_vals.quadrant_disable = 0;
	}
      }

    /* Get temperature (value or index) */
    } else if (lp != NULL && strcmp(wp[0], "temperature") == 0) {
      /* If only 2 words, it's the temp index */
      if (wc == 2) {
	temp_index = atoi(wp[1]);
	if (temp_index < 0 || temp_index >= LF_SWITCH_NUM_TEMPS) {
	  LF_ERROR(("Error: temp index is %d for slot %d",
		temp_index, lf_slot_display_no(lp)));
	}

      /* If 3 words, it's the temperature value */
      } else if (wc == 3) {
	ldp->new_vals.temperature[temp_index] = atoi(wp[1]);
      }

    /* Get overtemp count */
    } else if (lp != NULL && strcmp(wp[0], "overtemps") == 0) {
      ldp->new_vals.overtempcount = atoi(wp[1]);

    }

    /* This section is where we get data values and save them away */


    /* xbar values */

    PARSE_VALUE(xdp, "good", "crcs", NULL, 2, goodcrcs)
    PARSE_VALUE(xdp, "bad", "crcs", NULL, 2, badcrcs)
    PARSE_VALUE(xdp, "invalid", "routes", NULL, 2, invalidroutes)
    PARSE_VALUE(xdp, "receive", "timeout", "count", 3, receivetimeoutcount)
    PARSE_VALUE(xdp, "transmit", "timeout", "count", 3, transmittimeoutcount)
    PARSE_VALUE(xdp, "port", "down", NULL, 2, portdown)
    PARSE_VALUE(xdp, "port", "down", "count", 3, portflipcount)

    /* transceiver values */

    PARSE_VALUE(xcdp, "signal", "lost", NULL, 2, signallost)
    PARSE_VALUE(xcdp, "signal", "lost", "count", 3, signallostcount)

    /* "control" is common to xbars ports and fiber ports */
    else if (strcmp(wp[0], "control") == 0) {
      if (port_is_xbar) {
	if (xdp == NULL) LF_ERROR(("Unexpected keyword %s", wp[1]));
	xdp->new_vals.control = atoi(wp[1]);
      } else {
	if (xcdp == NULL) LF_ERROR(("Unexpected keyword %s", wp[1]));
	xcdp->new_vals.control = atoi(wp[1]);
      }
    }

    /* report delayed error */
    if (badstr != NULL) LF_ERROR(("Unexpected keyword %s", badstr));
  }

  lf_wget_close(fp);
  fp = NULL;

  /* make sure we got at least SOMETHING */
  if (userslot == -1) {
    LF_ERROR(("No data received from switch %s", ep->name));
  }

  return LF_QUERY_OK;

 except:
  if (fp != NULL) lf_wget_close(fp);
  return LF_QUERY_FATAL;
}

/*
 * Determine the switch type and fill in switch info if it is
 * an x32 based switch.
 */
int
lf_probe_switch_x32(
  struct lf_enclosure *ep)
{
  FILE *fp;
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  char *rv;
  int rc;
  int wc;
  int bp_type;
  struct sw32_encl_desc *eip;

  /* init for cleanup */
  fp = NULL;

  /*
   * First, make a pass which will tell us the enclosure type
   * and whether this is an x32 switch at all
   * Find the enclosure type based on backplane type.
   */
  fp = lf_wget_open(ep->name, "/cgi/web.cgi?slot.m");
  if (fp == NULL) {
    return -1;
  }

  /*
   * first line should be "Myricom Switch"
   */
  rv = lf_wget_next_line(fp, buf, wp, &wc);
  if (rv == NULL
      || wc < 2
      || strcmp(wp[0], "Myricom") != 0
      || strcmp(wp[1], "Switch") != 0) {
    goto except;
  }

  /*
   * OK, we know it's an x32 switch, look for backplane type
   */
  bp_type = -1;
  while (lf_wget_next_line(fp, buf, wp, &wc) != NULL) {
    if (wc == 3
	&& strcmp(wp[0], "backplane") == 0
	&& strcmp(wp[1], "type") == 0) {
      bp_type = atoi(wp[2]);
      break;
    }
  }
  if (bp_type == -1) {
    LF_ERROR(("Error finding backplane type for %s", ep->name));
  }
  lf_wget_close(fp);
  fp = NULL;
	
  /* convert backplane type to a product id */
  eip = sw32_encl_info;
  while (eip->product_id != NULL) {
    if (eip->bp_type == bp_type) {
      break;
    }
    ++eip;
  }
  if (eip->product_id == NULL) {
    LF_ERROR(("bad backplane type (%d)", bp_type));
  }

  /*
   * Fill in enclosure struct based on its type
   */
  LF_DUP_STRING(ep->product_id, eip->product_id);
  rc = lf_fill_enclosure(ep);
  if (rc == -1) goto except;

  fp = NULL;

  return 0;

 except:
  if (fp != NULL) lf_wget_close(fp);
  return -1;
}

/*
 * We found a new linecard - find the type
 */
static int
note_new_x32_linecard(
  FILE *fp,
  struct lf_enclosure *ep,
  int slot,
  char *product_id)
{
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  char *serial_no;
  int wc;

  serial_no = NULL;
  while (lf_wget_next_line(fp, buf, wp, &wc) != NULL) {
    if (wc > 1
	&& strcmp(wp[0], "serial") == 0
	&& strcmp(wp[1], "number") == 0) {
      if (wc > 2) {
	serial_no = wp[2];
      }
      product_id = lf_product_id_alias(product_id);
      return lf_note_new_linecard(ep, slot, product_id, serial_no);
    }
  }

  return -1;
}
